<?php

namespace App\Models\api\v1;

use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;

/**
 * @property integer $id
 * @property integer $server_id
 * @property string $code
 * @property float $amount_due
 * @property float $rebate
 * @property float $remaining_amount
 * @property integer $order_status_id
 * @property boolean $is_active
 * @property string $created_at
 * @property integer $created_by
 * @property integer $updated_by
 * @property string $updated_at
 * @property boolean $is_deleted
 * @property string $deleted_at
 * @property integer $deleted_by
 * @property User $user
 * @property LineItem[] $lineItems
 * @property Transaction[] $transactions
 */
class Order extends Model
{
    /**
     * The "type" of the auto-incrementing ID.
     * 
     * @var string
     */
    protected $keyType = 'integer';

    /**
     * @var array
     */
    protected $fillable = ['server_id', 'uuid', 'amount_due', 'rebate', 'remaining_amount', 'order_status_id',
        'is_active', 'payment_method', 'created_at', 'created_by', 'updated_by', 'updated_at', 'is_deleted', 'deleted_at',
        'deleted_by', 'table_number'];
    protected $hidden = [ 'created_by', 'updated_by', 'updated_at', 'is_deleted', 'deleted_at', 'deleted_by'];


    public static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            $model->uuid = Str::uuid();
            $model->created_by = auth()->user()->id ??  null;
            $model->updated_by = auth()->user()->id ??  null;
            $model->created_at = now();
            // $model->reference = Str::random(12);
        });

        static::updating(function ($model) {
            $model->updated_by = auth()->user()->id ??  null;
            $model->updated_at = now();
            // $model->reference = Str::random(12);
        });

    }
    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo('App\Models\api\v1\User', 'server_id');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lineItems()
    {
        return $this->hasMany('App\Models\api\v1\LineItem');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function transactions()
    {
        return $this->hasMany('App\Models\api\v1\Transaction');
    }

        /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function orderStatus()
    {
        return $this->belongsTo(OrderStatus::class);
    }

    public static function getOrdersForSpace(int $space_id, int $step_number=1): array
    {
        return Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus','lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $space_id]])
            ->where([['order_statuses.step_number', '>', $step_number]])
            ->orderBy('orders.created_at','asc')
            ->distinct()
            ->get()
            ->toArray();
    }
    public static function getOrdersForSpaceByStatus(int $space_id, int $order_status_id,int $step_number=1): array
    {
        return Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus','lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([['stocks.space_id', $space_id]])
            ->where([['orders.order_status_id', $order_status_id]])
            //->where([['order_statuses.step_number', '>', $step_number]])
            ->orderBy('orders.created_at','asc')
            ->distinct()
            ->get()
            ->toArray();
    }
}
